﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/SortOrder.h>
#include <aws/sagemaker/model/SortTrialComponentsBy.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class ListTrialComponentsRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API ListTrialComponentsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListTrialComponents"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A filter that returns only components that are part of the specified
   * experiment. If you specify <code>ExperimentName</code>, you can't filter by
   * <code>SourceArn</code> or <code>TrialName</code>.</p>
   */
  inline const Aws::String& GetExperimentName() const { return m_experimentName; }
  inline bool ExperimentNameHasBeenSet() const { return m_experimentNameHasBeenSet; }
  template <typename ExperimentNameT = Aws::String>
  void SetExperimentName(ExperimentNameT&& value) {
    m_experimentNameHasBeenSet = true;
    m_experimentName = std::forward<ExperimentNameT>(value);
  }
  template <typename ExperimentNameT = Aws::String>
  ListTrialComponentsRequest& WithExperimentName(ExperimentNameT&& value) {
    SetExperimentName(std::forward<ExperimentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only components that are part of the specified trial.
   * If you specify <code>TrialName</code>, you can't filter by
   * <code>ExperimentName</code> or <code>SourceArn</code>.</p>
   */
  inline const Aws::String& GetTrialName() const { return m_trialName; }
  inline bool TrialNameHasBeenSet() const { return m_trialNameHasBeenSet; }
  template <typename TrialNameT = Aws::String>
  void SetTrialName(TrialNameT&& value) {
    m_trialNameHasBeenSet = true;
    m_trialName = std::forward<TrialNameT>(value);
  }
  template <typename TrialNameT = Aws::String>
  ListTrialComponentsRequest& WithTrialName(TrialNameT&& value) {
    SetTrialName(std::forward<TrialNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only components that have the specified source Amazon
   * Resource Name (ARN). If you specify <code>SourceArn</code>, you can't filter by
   * <code>ExperimentName</code> or <code>TrialName</code>.</p>
   */
  inline const Aws::String& GetSourceArn() const { return m_sourceArn; }
  inline bool SourceArnHasBeenSet() const { return m_sourceArnHasBeenSet; }
  template <typename SourceArnT = Aws::String>
  void SetSourceArn(SourceArnT&& value) {
    m_sourceArnHasBeenSet = true;
    m_sourceArn = std::forward<SourceArnT>(value);
  }
  template <typename SourceArnT = Aws::String>
  ListTrialComponentsRequest& WithSourceArn(SourceArnT&& value) {
    SetSourceArn(std::forward<SourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only components created after the specified time.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAfter() const { return m_createdAfter; }
  inline bool CreatedAfterHasBeenSet() const { return m_createdAfterHasBeenSet; }
  template <typename CreatedAfterT = Aws::Utils::DateTime>
  void SetCreatedAfter(CreatedAfterT&& value) {
    m_createdAfterHasBeenSet = true;
    m_createdAfter = std::forward<CreatedAfterT>(value);
  }
  template <typename CreatedAfterT = Aws::Utils::DateTime>
  ListTrialComponentsRequest& WithCreatedAfter(CreatedAfterT&& value) {
    SetCreatedAfter(std::forward<CreatedAfterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only components created before the specified time.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedBefore() const { return m_createdBefore; }
  inline bool CreatedBeforeHasBeenSet() const { return m_createdBeforeHasBeenSet; }
  template <typename CreatedBeforeT = Aws::Utils::DateTime>
  void SetCreatedBefore(CreatedBeforeT&& value) {
    m_createdBeforeHasBeenSet = true;
    m_createdBefore = std::forward<CreatedBeforeT>(value);
  }
  template <typename CreatedBeforeT = Aws::Utils::DateTime>
  ListTrialComponentsRequest& WithCreatedBefore(CreatedBeforeT&& value) {
    SetCreatedBefore(std::forward<CreatedBeforeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The property used to sort results. The default value is
   * <code>CreationTime</code>.</p>
   */
  inline SortTrialComponentsBy GetSortBy() const { return m_sortBy; }
  inline bool SortByHasBeenSet() const { return m_sortByHasBeenSet; }
  inline void SetSortBy(SortTrialComponentsBy value) {
    m_sortByHasBeenSet = true;
    m_sortBy = value;
  }
  inline ListTrialComponentsRequest& WithSortBy(SortTrialComponentsBy value) {
    SetSortBy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sort order. The default value is <code>Descending</code>.</p>
   */
  inline SortOrder GetSortOrder() const { return m_sortOrder; }
  inline bool SortOrderHasBeenSet() const { return m_sortOrderHasBeenSet; }
  inline void SetSortOrder(SortOrder value) {
    m_sortOrderHasBeenSet = true;
    m_sortOrder = value;
  }
  inline ListTrialComponentsRequest& WithSortOrder(SortOrder value) {
    SetSortOrder(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of components to return in the response. The default value
   * is 10.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListTrialComponentsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If the previous call to <code>ListTrialComponents</code> didn't return the
   * full set of components, the call returns a token for getting the next set of
   * components.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListTrialComponentsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_experimentName;

  Aws::String m_trialName;

  Aws::String m_sourceArn;

  Aws::Utils::DateTime m_createdAfter{};

  Aws::Utils::DateTime m_createdBefore{};

  SortTrialComponentsBy m_sortBy{SortTrialComponentsBy::NOT_SET};

  SortOrder m_sortOrder{SortOrder::NOT_SET};

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_experimentNameHasBeenSet = false;
  bool m_trialNameHasBeenSet = false;
  bool m_sourceArnHasBeenSet = false;
  bool m_createdAfterHasBeenSet = false;
  bool m_createdBeforeHasBeenSet = false;
  bool m_sortByHasBeenSet = false;
  bool m_sortOrderHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
